//*************************************************************************************************
//
//	Description:
//		Snowcover.fx - Snowcover object shader for The Race. A specialisation of basic.fx with
//									 an extra layer which blends in according to the orientation of faces.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     02/07/2007  0.1           Created
//		TNettleship			02/12/2007	0.2						Fixed environment mapping
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//		TNettleship			10/23/2007	2.04					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(_PS3_)
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

float snowThreshold														// Threshold below which snow stops (vector component, not an angle)
<
	string UIName = "Snow threshold";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.5f;

float snowFade																// Vector amount during which snow fades in
<
	string UIName = "Snow fade";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.1f;

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

bool useNormalMap															// TRUE if the normal map is to be used in lighting
<
	string UIName = "Use normal map?";
	bool appEdit = true;
> = false;

texture normalTexture : TEXTURE								// Normal map + specularity factor in alpha.
<
	string UIName = "Normal Texture";
	bool appEdit = true;
>;

bool useEnvironmentMap												// TRUE if the environment map is to be used
<
	string UIName = "Use environment map?";
	bool appEdit = true;
> = false;

texture environmentTexture : TEXTURE					// Environment map (RGB)
<
	string UIName = "Env Texture";
	string Type = "CUBE";
	bool appEdit = false;
	bool export = false;
>;

texture snowDiffuse : TEXTURE
<
	string UIName = "Snow Diffuse";
	bool appEdit = true;
>;

texture snowSpecular : TEXTURE
<
	string UIName = "Snow Specular";
	bool appEdit = true;
>;

texture snowNormal : TEXTURE
<
	string UIName = "Snow Normal";
	bool appEdit = true;
>;

float minSpec
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpec
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpec
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float globalEMap
<
	string UIName = "EMap Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float snowMinSpec
<
	string UIName = "Snow Min Specular";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float snowMaxSpec
<
	string UIName = "Snow Max Specular";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float snowGlobalSpecular
<
	string UIName = "Snow Global Specular";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float snowGlobalEMap
<
	string UIName = "Snow Global Envmap";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D snowDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="snowDiffuse"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < snowDiffuse >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D snowSpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="snowSpecular"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < snowSpecular >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D snowNormalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="snowNormal"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < snowNormal >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float3 texCoordPlusSnowFactor		: TEXCOORD0;					// UV coords for texture channel 0 + snow factor in z
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT SnowcoverVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoordPlusSnowFactor.xy = _input.texCoord;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;
	
	// Calculate snow factor (rubbish but clear)
#ifdef _3DSMAX_
	float upwardNormalComponent = normal.z;
#else
	float upwardNormalComponent = normal.y;
#endif

	float snowFactor;
	if ( upwardNormalComponent < snowThreshold )
	{
		snowFactor = 0.0f;
	}
	else
	{
		if ( upwardNormalComponent < snowThreshold + snowFade )
		{
			snowFactor = ( upwardNormalComponent - snowThreshold ) / max( snowFade, 0.01f );
		}
		else
		{
			snowFactor = 1.0f;
		}
	}
	_output.texCoordPlusSnowFactor.z = snowFactor;

	// Calculate object-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float3 texCoordPlusSnowFactor		: TEXCOORD0;					// UV coords for texture channel 0 + snow factor in z
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float3 texCoordPlusSnowFactor		: TEXCOORD0;					// UV coords for texture channel 0 + snow factor in z
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

	float3 tangent		: TEXCOORD4_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD5_centroid;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT SnowcoverFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float2 texCoord = float2( _input.texCoordPlusSnowFactor.xy );

	// Read base textures
	float4 baseDiffuseTexColour = tex2D( diffuseMap, texCoord );
	float4 baseSpecularTexColour = tex2D( specularMap, texCoord );

	float4 snowDiffuseColour = tex2D( snowDiffuseMap, texCoord );
	float4 snowSpecularColour = tex2D( snowSpecularMap, texCoord );

	// Calculate interpolated textures
	float4 diffuseTexColour = lerp( baseDiffuseTexColour, snowDiffuseColour, _input.texCoordPlusSnowFactor.z );
	float4 specularTexColour = lerp( baseSpecularTexColour, snowSpecularColour, _input.texCoordPlusSnowFactor.z );

	// Calculate various interpolated factors
	float minSpecPowerValue = lerp( minSpec, snowMinSpec, _input.texCoordPlusSnowFactor.z );
	float maxSpecPowerValue = lerp( maxSpec, snowMaxSpec, _input.texCoordPlusSnowFactor.z );
	float globalSpecularFactorValue = lerp( globalSpec, snowGlobalSpecular, _input.texCoordPlusSnowFactor.z );
	float globalEMapFactor = lerp( globalEMap, snowGlobalEMap, _input.texCoordPlusSnowFactor.z );

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
	float3 tangent = normalize( _input.tangent );
	float3 binormal = normalize( _input.binormal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// If normal map support is required
	if ( useNormalMap )
	{
		// Fetch and decode the map normal
		float4 baseNormalMapColour = tex2D( normalMap, texCoord );
		float4 snowNormalMapColour = tex2D( snowNormalMap, texCoord );
		float4 normalMapColour = lerp( baseNormalMapColour, snowNormalMapColour, _input.texCoordPlusSnowFactor.z );

		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
	}

	// No normal map, so use interpolated normal and constant specular strength
	else
	{
		normal = TSnormal;
	}

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// If environment mapping is switched on
	if ( useEnvironmentMap )
	{
		float3 envMapSpaceNormal = normal;
		float3 envMapSpaceEye = -eye;
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;
	}

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}




//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float3 texCoordPlusSnowFactor		: TEXCOORD0;					// UV coords for texture channel 0 + snow factor in z
};



VSOUTPUT_LD SnowcoverLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoordPlusSnowFactor.xy = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate snow factor (rubbish but clear)
#ifdef _3DSMAX_
	float upwardNormalComponent = normal.z;
#else
	float upwardNormalComponent = normal.y;
#endif

	float snowFactor;
	if ( upwardNormalComponent < snowThreshold )
	{
		snowFactor = 0.0f;
	}
	else
	{
		if ( upwardNormalComponent < snowThreshold + snowFade )
		{
			snowFactor = ( upwardNormalComponent - snowThreshold ) / max( snowFade, 0.01f );
		}
		else
		{
			snowFactor = 1.0f;
		}
	}
	_output.texCoordPlusSnowFactor.z = snowFactor;

	_output.colour = _input.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}


PSOUTPUT SnowcoverLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float2 texCoord = float2( _input.texCoordPlusSnowFactor.xy );

	// Read base textures
	float4 baseDiffuseTexColour = tex2D( diffuseMap, texCoord );

	float4 snowDiffuseColour = tex2D( snowDiffuseMap, texCoord );

	// Calculate interpolated textures
	float4 diffuseTexColour = lerp( baseDiffuseTexColour, snowDiffuseColour, _input.texCoordPlusSnowFactor.z );


	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Snowcover
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Snowcover";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "SnowCover_LowDetail";
	int    lowDetailDeferredID		= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx SnowcoverVertexShader();
		PixelShader = compile sce_fp_rsx SnowcoverFragmentShader();
#else
		VertexShader = compile vs_3_0 SnowcoverVertexShader();
		PixelShader = compile ps_3_0 SnowcoverFragmentShader();
#endif
	}
}



technique Snowcover_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Snowcover_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx SnowcoverLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx SnowcoverLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 SnowcoverLowDetailVertexShader();
		PixelShader = compile ps_3_0 SnowcoverLowDetailFragmentShader();
#endif
	}
}
